<?php
   
    if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

   
    /**
     * Core class used to implement displaying sites in a list table for the network admin.
     *
     * @since 3.1.0
     * @access private
     *
     * @see WP_List_Table
     */
    class WP_Domains_List_Table extends WP_List_Table {

	    /**
	     * Site status list.
	     *
	     * @since 4.3.0
	     * @var array
	     */
	    public $status_list;
        
        public $functions;
        
        public $woomd_network;

	    /**
	     * Constructor.
	     *
	     * @since 3.1.0
	     *
	     * @see WP_List_Table::__construct() for more information on default arguments.
	     *
	     * @param array $args An associative array of arguments.
	     */
	    public function __construct( $args = array() ) 
            {
		  
		        parent::__construct( array(
			        'plural' => 'domains',
			        'screen' => isset( $args['screen'] ) ? $args['screen'] : null,
		        ) );
                
                global $woo_multidomain;
                $this->functions        =   $woo_multidomain->functions;
                $this->woomd_network    =   $woo_multidomain->functions->network;
                
	        }


	    /**
	     * Prepares the list of sites for display.
	     *
	     * @since 3.1.0
	     *
	     * @global string $s
	     * @global string $mode
	     * @global wpdb   $wpdb
	     */
	    public function prepare_items() 
            {
		        global $s, $mode, $wpdb;

		        if ( ! empty( $_REQUEST['mode'] ) ) 
                    {
			            $mode = $_REQUEST['mode'] === 'excerpt' ? 'excerpt' : 'list';
			            set_user_setting( 'sites_list_mode', $mode );
		            } 
                else 
                    {
			            $mode = get_user_setting( 'sites_list_mode', 'list' );
		            }

		        $per_page = $this->get_items_per_page( 'woomd_domains_per_page' );

		        $pagenum = $this->get_pagenum();

		        $s = isset( $_REQUEST['s'] ) ? wp_unslash( trim( $_REQUEST[ 's' ] ) ) : '';
		        $wild = '';
		        if ( false !== strpos($s, '*') ) {
			        $wild = '*';
			        $s = trim($s, '*');
		        }


		        $args = array(
			        'number'     => intval( $per_page ),
			        'offset'     => intval( ( $pagenum - 1 ) * $per_page )
		        );

		        if ( empty($s) ) 
                    {
			            // Nothing to do.
		            } 
                    elseif ( preg_match( '/^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$/', $s ) ||
					            preg_match( '/^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.?$/', $s ) ||
					            preg_match( '/^[0-9]{1,3}\.[0-9]{1,3}\.?$/', $s ) ||
					            preg_match( '/^[0-9]{1,3}\.$/', $s ) ) 
                                {
			            // IPv4 address
			            $sql = $wpdb->prepare( "SELECT blog_id FROM {$wpdb->registration_log} WHERE {$wpdb->registration_log}.IP LIKE %s", $wpdb->esc_like( $s ) . ( ! empty( $wild ) ? '%' : '' ) );
			            $reg_blog_ids = $wpdb->get_col( $sql );

			            if ( $reg_blog_ids ) {
				            $args['site__in'] = $reg_blog_ids;
			            }
		            } elseif ( is_numeric( $s ) && empty( $wild ) ) 
                    {
			            $args['ID'] = $s;
		            } else 
                    {
			            $args['search'] = $s;
		            }

		        $order_by = isset( $_REQUEST['orderby'] ) ? $_REQUEST['orderby'] : '';
		        if ( 'registered' === $order_by ) {
			        // registered is a valid field name.
		        } elseif ( 'blogname' === $order_by ) {
			        $order_by = 'domain';
		        } elseif ( ! $order_by ) {
			        $order_by = '';
		        }

		        $args['orderby'] = $order_by;

		        if ( ! empty ($order_by )  )
                    {
			            $args['order'] = ( isset( $_REQUEST['order'] ) && 'DESC' === strtoupper( $_REQUEST['order'] ) ) ? "DESC" : "ASC";
		            }

		        $args['no_found_rows'] = false;

		        /**
		         * Filters the arguments for the site query in the sites list table.
		         *
		         * @since 4.6.0
		         *
		         * @param array $args An array of get_sites() arguments.
		         */
		        $args = apply_filters( 'woomd/domains_list_table_query_args', $args );

		        $data           =   $this->functions->network->get_domains( $args );
                $this->items    =   $data['items'];
                
		        $total_domains  =   $data['total_founds'];

		        $this->set_pagination_args( array(
			                                        'total_items' => $total_domains,
			                                        'per_page' => $per_page,
		                                        ) );
                
                
                $columns    = $this->get_columns();
                $hidden     = array();
                $sortable   = $this->get_sortable_columns();
                
                $this->_column_headers = array($columns, $hidden, $sortable);
	        }


	    /**
	     *
	     * @return array
	     */
	    protected function get_bulk_actions() 
            {
		        $actions = array();
		        $actions['delete'] = __( 'Delete' );
		        
		        return $actions;
	        }

	    /**
	     * @global string $mode List table view mode.
	     *
	     * @param string $which
	     */
	    protected function pagination( $which ) 
            {
		        global $mode;

		        parent::pagination( $which );

		        if ( 'top' === $which )
			        $this->view_switcher( $mode );
	        }

	    /**
	     * @return array
	     */
	    public function get_columns() 
            {
		        $sites_columns = array(
			                                'cb'          => '<input type="checkbox" />',
			                                'blogname'    => __( 'Domain' ),
			                                'registered'  => _x( 'Registered', 'site' ),
			                                'users'       => __( 'Users' ),
		                                );
       
		        return apply_filters( 'woomd/wpmu_blogs_columns', $sites_columns );
	        }

	    /**
	     * @return array
	     */
	    protected function get_sortable_columns() 
            {
		        return array(
			        'blogname'    =>  array ('domain', TRUE ),
			        'registered'  =>  array ( 'registered', TRUE )
		        );
	        }

	    /**
	     * Handles the checkbox column output.
	     *
	     * @since 4.3.0
	     *
	     * @param array $blog Current site.
	     */
	    public function column_cb( $domain ) 
            {
		                    
                if ( $domain['id']   !=  1 ) 
                    {
			            $blogname = untrailingslashit( $domain['domain'] . $domain['path'] );
		                ?>
			                <label class="screen-reader-text" for="blog_<?php echo $domain['id']; ?>"><?php
				                printf( __( 'Select %s' ), $blogname );
			                ?></label>
			                <input type="checkbox" id="blog_<?php echo $domain['id'] ?>" name="allblogs[]" value="<?php echo esc_attr( $domain['id'] ) ?>" />
		                <?php 
                    }
	        }

	    /**
	     * Handles the ID column output.
	     *
	     * @since 4.4.0
	     *
	     * @param array $blog Current site.
	     */
	    public function column_id( $blog ) 
            {
		        echo $blog['id'];
	        }

	    /**
	     * Handles the site name column output.
	     *
	     * @since 4.3.0
	     *
	     * @global string $mode List table view mode.
	     *
	     * @param array $blog Current site.
	     */
	    public function column_blogname( $blog ) 
            {
		        global $mode;

		        $blogname       =   untrailingslashit( $blog['domain'] . $blog['path'] );
		  
		        ?>
		        <strong>
			        <a href="<?php echo esc_url( network_admin_url( 'site-info.php?id=' . $blog['id'] ) ); ?>" class="edit"><?php echo $blogname; ?></a>
		        </strong>
		        <?php
		        if ( 'list' !== $mode ) 
                    {
			            $this->woomd_network->switch_to_domain( $blog['id'] );
                        
			            echo '<p>';
			            printf(
				            /* translators: 1: site name, 2: site tagline. */
				            __( '%1$s &#8211; %2$s' ),
				            get_option( 'blogname' ),
				            '<em>' . get_option( 'blogdescription ' ) . '</em>'
			            );
			            echo '</p>';
			            
                        $this->woomd_network->restore_current_domain();
		            }
	        }

	    /**
	     * Handles the registered column output.
	     *
	     * @since 4.3.0
	     *
	     * @global string $mode List table view mode.
	     *
	     * @param array $blog Current site.
	     */
	    public function column_registered( $blog ) {
		    global $mode;

		    if ( 'list' === $mode ) {
			    $date = __( 'Y/m/d' );
		    } else {
			    $date = __( 'Y/m/d g:i:s a' );
		    }

		    if ( $blog['registered'] === '0000-00-00 00:00:00' ) {
			    echo '&#x2014;';
		    } else {
			    echo mysql2date( $date, $blog['registered'] );
		    }
	    }

	    /**
	     * Handles the users column output.
	     *
	     * @since 4.3.0
	     *
	     * @param array $blog Current site.
	     */
	    public function column_users( $blog ) {
		    $user_count = wp_cache_get( $blog['id'] . '_user_count', 'blog-details' );
		    if ( ! $user_count ) {
			    $blog_users = get_users( array( 'id' => $blog['id'], 'fields' => 'ID' ) );
			    $user_count = count( $blog_users );
			    unset( $blog_users );
			    wp_cache_set( $blog['id'] . '_user_count', $user_count, 'blog-details', 12 * HOUR_IN_SECONDS );
		    }

		    printf(
			    '<a href="%s">%s</a>',
			    esc_url( network_admin_url( 'users.php?id=' . $blog['id'] ) ),
			    number_format_i18n( $user_count )
		    );
	    }

	    /**
	     * Handles output for the default column.
	     *
	     * @since 4.3.0
	     *
	     * @param array  $blog        Current site.
	     * @param string $column_name Current column name.
	     */
	    public function column_default( $blog, $column_name ) {
		    /**
		     * Fires for each registered custom column in the Sites list table.
		     *
		     * @since 3.1.0
		     *
		     * @param string $column_name The name of the column to display.
		     * @param int    $blog_id     The site ID.
		     */
		    do_action( 'manage_sites_custom_column', $column_name, $blog['id'] );
	    }

	    /**
	     *
	     * @global string $mode
	     */
	    public function display_rows() 
            {
		        foreach ( $this->items as $domain ) {
			       
			        $class = '';
			 
			        echo "<tr{$class}>";

			        $this->single_row_columns( $domain );

			        echo '</tr>';
		        }
	        }

	    /**
	     * Gets the name of the default primary column.
	     *
	     * @since 4.3.0
	     *
	     * @return string Name of the default primary column, in this case, 'blogname'.
	     */
	    protected function get_default_primary_column_name() {
		    return 'blogname';
	    }

	    /**
	     * Generates and displays row action links.
	     *
	     * @since 4.3.0
	     *
	     * @param object $blog        Site being acted upon.
	     * @param string $column_name Current column name.
	     * @param string $primary     Primary column name.
	     * @return string Row actions output.
	     */
	    protected function handle_row_actions( $blog, $column_name, $primary ) {
		    if ( $primary !== $column_name ) {
			    return;
		    }
            
		    $blogname = untrailingslashit( $blog['domain'] . $blog['path'] );

		    // Preordered.
		    $actions = array(
			    'edit' => '', 'backend' => '',

			    'delete' => '',
			    'visit' => '',
		    );

		    $actions['edit']	= '<a href="' . esc_url( network_admin_url( 'admin.php?page=woo-multidomain&section=domain-edit&id=' . $blog['id'] ) ) . '">' . __( 'Edit' ) . '</a>';
		    $actions['backend']	= "<a href='" . esc_url( trailingslashit ( $blog['domain'] ) . 'wp-admin/' ) . "' class='edit'>" . __( 'Dashboard' ) . '</a>';
           
            if ( $blog['id'] != 1 ) 
	            $actions['delete'] = '<a href="' . esc_url( wp_nonce_url( get_admin_url( null,  'admin.php?page=woo-multidomain&action=confirm&amp;action2=deleteblog&amp;id=' . $blog['id'] ), 'deleteblog_' . $blog['id'] ) ) . '">' . __( 'Delete' ) . '</a>';

		    $actions['visit']	= "<a href='" . esc_url( trailingslashit ( $blog['domain'] ) ) . "' rel='bookmark'>" . __( 'Visit' ) . '</a>';

		    $actions = apply_filters( 'woomd/manage_sites_action_links', array_filter( $actions ), $blog['id'], $blogname );
		    
            return $this->row_actions( $actions );
	    }
    }

    
    
?>