<?php


use Automattic\Jetpack\Constants;

defined( 'ABSPATH' ) || exit;

/**
 * Handle data for the current customers session.
 * Implements the WC_Session abstract class.
 *
 * From 2.5 this uses a custom table for session storage. Based on https://github.com/kloon/woocommerce-large-sessions.
 *
 * @class    WC_Session_Handler
 * @version  2.5.0
 * @package  WooCommerce/Classes
 * @category Class
 * @author   WooThemes
 */
class WooMD_WC_Session_Handler extends WC_Session_Handler {

	/** @var string cookie name */
	protected $_cookie;

	/** @var string session due to expire timestamp */
	protected $_session_expiring;

	/** @var string session expiration timestamp */
	protected $_session_expiration;

	/** $var bool Bool based on whether a cookie exists **/
	protected $_has_cookie = false;

	/** @var string Custom session table name */
	protected $_table;
    
    
    private $_woomd_use_session     =   TRUE;
    
    private $_woomd_session         =   FALSE;
    
    private $_woomd_session_refresh =   FALSE;
    
    private $_woomd_session_id;
    
    private $_woomd_cookies         =   array();
    
    private $_woomd_do_sync         =   FALSE;
        
    private $_woomd_table;

	/**
	 * Constructor for the session class.
	 */
    public function __construct() {
        
        parent::__construct();
        
        $this->_woomd_cookies['cookie']            =   'woomd_session';
        $this->_woomd_cookies['cookie_trigger']    =   'woomd_sync_run';
        $this->_woomd_table                         =   $GLOBALS['wpdb']->base_prefix . 'woocommerce_woomd_sessions';
        
        $options    =   Woo_MultiDomain_Functions::get_options();
        global $blog_id;
        
        if ( isset ( $options['use_global_cart_for_sites'][$blog_id] )    &&  $options['use_global_cart_for_sites'][$blog_id] == 'no' )
            $this->_woomd_use_session  =   FALSE;
        
    }
    

    /**
     * Setup cookie and customer ID.
     *
     * @since 3.6.0
     */
    public function init_session_cookie() {
        $cookie = $this->get_session_cookie();
        
        if ( $cookie ) {
            $this->_customer_id        = $cookie[0];
            $this->_session_expiration = $cookie[1];
            $this->_session_expiring   = $cookie[2];
            $this->_has_cookie         = true;
            $this->_data               = $this->get_session_data();

            // Update session if its close to expiring.
            if ( time() > $this->_session_expiring ) {
                $this->set_session_expiration();
                $this->update_session_timestamp( $this->_customer_id, $this->_session_expiration );
            }
                        
        } else {
            $this->set_session_expiration();
            $this->_customer_id = $this->generate_customer_id();
            $this->_data        = $this->get_session_data();
        }
        
        if ( $this->_woomd_use_session )
            $this->woomd_open_session();
    }

	/**
	 * Sets the session cookie on-demand (usually after adding an item to the cart).
	 *
	 * Since the cookie name (as of 2.1) is prepended with wp, cache systems like batcache will not cache pages when set.
	 *
	 * Warning: Cookies will only be set if this is called before the headers are sent.
	 */
	public function set_customer_session_cookie( $set ) {
		if ( $set ) {
            
            parent::set_customer_session_cookie( $set );
                        
            if ( $this->_woomd_use_session )
                {            
                    //save the woomd sesion if created in this load
                    if ( $this->_woomd_session  === FALSE )
                        {
                            $this->woomd_insert_session();
                            $this->_woomd_session       =   $this->woomd_get_session( $this->_woomd_session_id );
                        }
                        
                    if ( ! $this->has_woomd_session()   ||  $this->_woomd_session_refresh   === TRUE )
                        $this->woomd_set_cookies();
                }
        }
	}
 
    
    function woomd_open_session()
        {
            if ( $this->has_woomd_session() )
                {
                    $this->_woomd_session_id    =   preg_replace("/[^a-zA-Z0-9]/", "", $_COOKIE[ $this->_woomd_cookies['cookie'] ] );   
                    $this->_woomd_session       =   $this->woomd_get_session( $this->_woomd_session_id );
                        
                    if ( $this->_woomd_session === FALSE    ||  time() > $this->_woomd_session['session_expiry'] )
                        {
                            $this->woomd_delete_session( $this->_woomd_session_id );
                            $this->_woomd_session           =   FALSE;
                            $this->_woomd_session_refresh   =   TRUE;
                            $this->_woomd_session_id        =   $this->generate_woomd_session();
                            return FALSE;
                        }
                        
                    //check for about to expire
                    if ( ( time() + 60 *60 ) > $this->_woomd_session['session_expiry'] )
                        {
                            $this->woomd_delete_session( $this->_woomd_session_id );
                            $this->_woomd_session           =   FALSE;
                            $this->_woomd_session_refresh   =   TRUE;
                            $this->_woomd_session_id        =   $this->generate_woomd_session();
                            
                            $this->woomd_insert_session();
                            $this->_woomd_session       =   $this->woomd_get_session( $this->_woomd_session_id );
                        }
                        
                    //if ( ! empty  ( $this->_woomd_session['session_key'] )  &&  $this->session_exists( $this->_customer_id ) )
                    if ( ! empty  ( $this->_woomd_session['session_key'] )  &&  $this->session_exists( $this->_woomd_session['session_key'] ) )
                        $this->_customer_id = $this->_woomd_session['session_key'];
                    $this->_data        = $this->get_session( $this->_customer_id );
                 }
                else
                {
                    //create the session
                    $this->_woomd_session_id    =   $this->generate_woomd_session();
                }    
            
        }
    
    function woomd_set_cookies()
        {
            $this->_woomd_do_sync   =   TRUE;
            wc_setcookie( $this->_woomd_cookies['cookie_trigger'], $this->_woomd_session['trigger_key'], time() + WOOMD_TRIGGER_KEY_EXPIRE, $this->use_secure_cookie() );                                                                
            
            wc_setcookie( $this->_woomd_cookies['cookie'], $this->_woomd_session_id, time() + WOOMD_COOKIE_EXPIRE, $this->use_secure_cookie(), TRUE );
        }
        
        
    public function woomd_get_do_sync_status()
        {
            return  $this->_woomd_do_sync;
        }

    function generate_woomd_session( $woomd_session_key = '' )
        {
            if ( empty ($woomd_session_key ) )
                $woomd_session_key      =   md5 ( microtime() .rand( 1,999999 ) );
                        
            return $woomd_session_key;
        }
      

	/**
	 * Return true if the current user has an active session, i.e. a cookie to retrieve values.
	 *
	 * @return bool
	 */
	public function has_session() {
		return isset( $_COOKIE[ $this->_cookie ] ) || $this->_has_cookie;
	}
    
    
    public function has_woomd_session() {
        return isset( $_COOKIE[ $this->_woomd_cookies['cookie'] ] );
    }


	/**
	 * Generate a unique customer ID for guests, or return user ID if logged in.
	 *
	 * Uses Portable PHP password hashing framework to generate a unique cryptographically strong ID.
	 *
	 * @return int|string
	 */
    public function generate_customer_id() {
        $customer_id = '';

        if ( empty( $customer_id ) ) {
            require_once ABSPATH . 'wp-includes/class-phpass.php';
            $hasher      = new PasswordHash( 8, false );
            $customer_id = md5( $hasher->get_random_bytes( 32 ) );
        }

        return $customer_id;
    }
    
    public function get_customer_id() {
        
        if ( !empty($this->_customer_id) )
            return $this->_customer_id;
            
        return '';
    }
    
    public function woomd_get_session_id() {
        
        if ( !empty( $this->_woomd_session_id ) )
            return $this->_woomd_session_id;
            
        return '';
    }


	/**
	 * Save data.
	 */
    public function save_data( $old_session_key = 0 ) {
        // Dirty if something changed - prevents saving nothing new.
        if ( $this->_dirty && $this->has_session() ) {
            
            parent::save_data( $old_session_key );
            
            
            if ( $this->_woomd_use_session )
                {
                    //Save the woomd seession
                    if ( $this->_woomd_session !== FALSE )
                        {
                            //update
                            $this->woomd_update_session_customer_id( $this->_customer_id );
                               
                            return;
                        }
                    
                    $this->woomd_insert_session();
                }
             
        }
    }
    
        
    function woomd_insert_session()
        {
            global $wpdb;
            
            $woomd_trigger_key      =   substr( md5 ( microtime() .rand( 1,999999 ) ), 0 , 12 );
            
            // Create the token
            $token_data = array(
                                    'ip'        =>  $_SERVER['REMOTE_ADDR'],
                                    'ua'        =>  isset($_SERVER['HTTP_USER_AGENT'])  ?   wp_unslash( $_SERVER['HTTP_USER_AGENT'] )   :   ''
                                );
            $session_hash       = wp_hash( serialize( $token_data ) );
            
            //store the session key
            $mysql_query    =   $wpdb->prepare(
                                                    "INSERT INTO {$this->_woomd_table} ( `session_key`, `woomd_session_key`, `session_expiry`, `trigger_key`, `trigger_key_expiry`, `trigger_user_hash` ) VALUES ( %s, %s, %s, %s, %s, %s )",
                                                    $this->_customer_id,
                                                    $this->_woomd_session_id,
                                                    time() + WOOMD_COOKIE_EXPIRE,
                                                    $woomd_trigger_key,
                                                    time() + WOOMD_TRIGGER_KEY_EXPIRE,
                                                    $session_hash );
            $wpdb->query ( $mysql_query );   
        }
        

        
    /**
     * Gets a cache prefix. This is used in session names so the entire cache can be invalidated with 1 function call.
     *
     * @return string
     */
    private function get_cache_prefix() {
        return WC_Cache_Helper::get_cache_prefix( WC_SESSION_CACHE_GROUP );
    }
    
        
	/**
	 * Destroy all session data.
	 */
    public function destroy_session() {
        parent::destroy_session();
        
        if ( $this->_woomd_use_session )
            $this->woomd_update_session_customer_id( '' );
    }



	/**
	 * Cleanup sessions.
	 */
    public function cleanup_sessions() {
        global $wpdb;

        parent::cleanup_sessions();
        
        $wpdb->query( $wpdb->prepare( "DELETE FROM $this->_woomd_table WHERE session_expiry < %d", time() + WOOMD_COOKIE_EXPIRE ) );
        
    }

	/**
	 * Returns the session.
	 *
	 * @param string $customer_id
	 * @param mixed $default
	 * @return string|array
	 */
	public function get_session( $customer_id, $default = false ) {
		global $wpdb;

        if ( Constants::is_defined( 'WP_SETUP_CONFIG' ) ) {
            return false;
        }

        $value  =   false;
        
        if ( false === $value ) {
            $value = $wpdb->get_var( $wpdb->prepare( "SELECT session_value FROM $this->_table WHERE session_key = %s", $customer_id ) ); // @codingStandardsIgnoreLine.

            if ( is_null( $value ) ) {
                $value = $default;
            }

            $cache_duration = $this->_session_expiration - time();
            if ( 0 < $cache_duration ) {
                wp_cache_add( $this->get_cache_prefix() . $customer_id, $value, WC_SESSION_CACHE_GROUP, $cache_duration );
            }
        }

        return maybe_unserialize( $value );
	}
    
    
    function session_exists( $customer_id )
        {
            global $wpdb;   
            
            $value = $wpdb->get_var( $wpdb->prepare( "SELECT session_value FROM $this->_table WHERE session_key = %s", $customer_id ) );
            
            if ( is_null( $value ) )
                return FALSE;
                
            return TRUE;
        }
        
    
    public function woomd_get_session ( $session_id, $default = false )
        {
            global $wpdb;

            $data = $wpdb->get_row( $wpdb->prepare( "SELECT *  FROM {$this->_woomd_table} WHERE woomd_session_key = %s", $session_id ), ARRAY_A );

            if ( is_null ( $data ) )
                return FALSE;
                
            return $data;
            
        }
        
   
    
    public function woomd_delete_session( $_woomd_session_id ) {
        global $wpdb;

        $wpdb->delete(
            $this->_woomd_table,
            array(
                'woomd_session_key' => $_woomd_session_id,
            )
        );
    }
    
    
    public function woomd_update_session_customer_id( $customer_id  =   ''  )
        {
            global $wpdb;

            $wpdb->update(
                $this->_woomd_table,
                array(
                    'session_key' => $customer_id,
                ),
                array(
                    'woomd_session_key' => $this->_woomd_session_id,
                )
            );   
            
            
        }

}
